/* Copyright (c) 1998-2013 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Thorsten Kukuk <kukuk@suse.de>, 1998.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <http://www.gnu.org/licenses/>.  */

/* getent: get entries from administrative database.  */

//#include <aliases.h>
#include <argp.h>
#include <ctype.h>
#include <error.h>
#include <grp.h>
#ifndef __CYGWIN__
#include <gshadow.h>
#include <libintl.h>
#endif
#include <locale.h>
#ifndef __CYGWIN__
#include <mcheck.h>
#endif
#include <netdb.h>
#include <pwd.h>
#ifndef __CYGWIN__
#include <shadow.h>
#endif
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#ifndef __CYGWIN__
#include <netinet/ether.h>
#endif
#include <netinet/in.h>
#include <sys/socket.h>

#ifndef __CYGWIN__
/* Get libc version number.  */
#include <version.h>
#else
#include <windows.h>
#include <wchar.h>
#include <sddl.h>
#include <sys/cygwin.h>

#define _(string) (string)
#define N_(string) (string)
#define textdomain(string)
#define gettext(string) (string)

#define fputc_unlocked fputc
#define fputs_unlocked fputs
#define putchar_unlocked putchar

#define PKGVERSION	"(Cygwin)"
#define VERSION		"2.18.90"	/* glibc version we took getent from. */
#define REPORT_BUGS_TO	"http://cygwin.com/problems.html"

# define stackinfo_alloca_round(l) (((l) + 15) & -16)
# define extend_alloca(buf, len, newlen) \
  (__typeof (buf)) ({ size_t __newlen = stackinfo_alloca_round (newlen);      \
                      char *__newbuf = alloca (__newlen);                   \
                      if (__newbuf + __newlen == (char *) buf)                \
                        len += __newlen;                                      \
                      else                                                    \
                        len = __newlen;                                       \
                      __newbuf; })

#endif /* !__CYGWIN__ */

#define PACKAGE _libc_intl_domainname

/* Name and version of program.  */
static void print_version (FILE *stream, struct argp_state *state);
void (*argp_program_version_hook) (FILE *, struct argp_state *) = print_version;

/* Short description of parameters.  */
static const char args_doc[] = N_("database [key ...]");

/* Supported options. */
static const struct argp_option args_options[] =
  {
    { "service", 's', N_("CONFIG"), 0, N_("Service configuration to be used") },
    { "no-idn", 'i', NULL, 0, N_("disable IDN encoding") },
#ifdef __CYGWIN__
    { "windows", 'w', NULL, 0, N_("Cygwin:Windows account mapping only (with passwd and group databases)") },
#endif
    { NULL, 0, NULL, 0, NULL },
  };

/* Short description of program.  */
static const char doc[] = N_("Get entries from administrative database.");

/* Prototype for option handler.  */
static error_t parse_option (int key, char *arg, struct argp_state *state);

/* Function to print some extra text in the help message.  */
static char *more_help (int key, const char *text, void *input);

/* Data structure to communicate with argp functions.  */
static struct argp argp =
  {
    args_options, parse_option, args_doc, doc, NULL, more_help
  };

/* Additional getaddrinfo flags for IDN encoding.  */
static int idn_flags = AI_IDN | AI_CANONIDN;

#ifdef __CYGWIN__
static int windows_only = 0;
#endif

/* Print the version information.  */
static void
print_version (FILE *stream, struct argp_state *state)
{
  fprintf (stream, "getent %s%s\n", PKGVERSION, VERSION);
  fprintf (stream, gettext ("\
Copyright (C) %s Free Software Foundation, Inc.\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\
"), "2013");
  fprintf (stream, gettext ("Written by %s.\n"), "Thorsten Kukuk");
}

#ifndef __CYGWIN__
/* This is for aliases */
static void
print_aliases (struct aliasent *alias)
{
  unsigned int i = 0;

  printf ("%s: ", alias->alias_name);
  for  (i = strlen (alias->alias_name); i < 14; ++i)
    fputs_unlocked (" ", stdout);

  for (i = 0; i < alias->alias_members_len; ++i)
    printf ("%s%s",
	    alias->alias_members [i],
	    i + 1 == alias->alias_members_len ? "\n" : ", ");
}

static int
aliases_keys (int number, char *key[])
{
  int result = 0;
  int i;
  struct aliasent *alias;

  if (number == 0)
    {
      setaliasent ();
      while ((alias = getaliasent ()) != NULL)
	print_aliases (alias);
      endaliasent ();
      return result;
    }

  for (i = 0; i < number; ++i)
    {
      alias = getaliasbyname (key[i]);

      if (alias == NULL)
	result = 2;
      else
	print_aliases (alias);
    }

  return result;
}

/* This is for ethers */
static int
ethers_keys (int number, char *key[])
{
  int result = 0;
  int i;

  if (number == 0)
    {
      fprintf (stderr, _("Enumeration not supported on %s\n"), "ethers");
      return 3;
    }

  for (i = 0; i < number; ++i)
    {
      struct ether_addr *ethp, eth;
      char buffer [1024], *p;

      ethp = ether_aton (key[i]);
      if (ethp != NULL)
	{
	  if (ether_ntohost (buffer, ethp))
	    {
	      result = 2;
	      continue;
	    }
	  p = buffer;
	}
      else
	{
	  if (ether_hostton (key[i], &eth))
	    {
	      result = 2;
	      continue;
	    }
	  p = key[i];
	  ethp = &eth;
	}
      printf ("%s %s\n", ether_ntoa (ethp), p);
    }

  return result;
}
#endif /* !__CYGWIN__ */

#ifdef __CYGWIN__
static PSID
get_sid_from_special_string (const char *key, PBOOL try_as_cygname)
{
  PSID sid = NULL;

  if (!strncmp (key, "U-", 2))
    {
      WCHAR wnam[4096];

      /* DNLEN is moot.  Domain names can now be twice as long. */
      WCHAR dom[32];
      DWORD dlen = 32;
      DWORD slen = SECURITY_MAX_SID_SIZE;
      SID_NAME_USE use;

      memset (wnam, 0, sizeof wnam);
      mbstowcs (wnam, key + 2, sizeof wnam / sizeof (wchar_t) - 1);
      sid = (PSID) LocalAlloc (LMEM_FIXED, SECURITY_MAX_SID_SIZE);
      if (sid && !LookupAccountNameW (NULL, wnam, sid, &slen,
				      dom, &dlen, &use))
	{
	  LocalFree (sid);
	  sid = NULL;
	}
      if (sid
	  && (use == SidTypeDomain || use == SidTypeComputer)
	  && wcslen (wnam) < sizeof wnam / sizeof (wchar_t) / 2 - 2)
	{
	  wcpcpy (wnam + wcslen (wnam) + 1, wnam);
	  wnam[wcslen (wnam)] = L'\\';
	  dlen = 32;
	  slen = SECURITY_MAX_SID_SIZE;
	  if (!LookupAccountNameW (NULL, wnam, sid, &slen,
				   dom, &dlen, &use))
	    {
	      LocalFree (sid);
	      sid = NULL;
	    }
	}
    }
  else if (!strncmp (key, "S-1-", 4))
    {
      *try_as_cygname = TRUE;
      ConvertStringSidToSidA (key, &sid);
    }
  return sid;
}

static void
print_windows_mapping (const char *cygname, unsigned long id,
		       const char *sid_string)
{
  PSID sid;
  WCHAR name[NAME_MAX + 1];
  DWORD nlen = NAME_MAX + 1;
  /* DNLEN is moot.  Domain names can now be twice as long. */
  WCHAR dom[32];
  DWORD dlen = 32;
  SID_NAME_USE use;

  if (!cygname || !sid_string)
    return;
  if (!ConvertStringSidToSidA (sid_string, &sid))
    return;
  if (LookupAccountSidW (NULL, sid, name, &nlen, dom, &dlen, &use))
    printf ("%s:%lu:%ls\\%ls:%s\n", cygname, id, dom, name, sid_string);
  LocalFree (sid);
}
#endif

/* This is for group */
static void
print_group (struct group *grp)
{
  unsigned int i = 0;

#ifdef __CYGWIN__
  /* Disable printing gid -1, the fallback invalid group called "????????" on
     Cygwin versions prior to the introduction of direct SAM/AD access. */
  if (grp->gr_gid == (gid_t) -1)
    return;
  if (windows_only)
    {
      print_windows_mapping (grp->gr_name, (unsigned long) grp->gr_gid,
			     grp->gr_passwd);
      return;
    }
#endif
  printf ("%s:%s:%lu:", grp->gr_name ? grp->gr_name : "",
	  grp->gr_passwd ? grp->gr_passwd : "",
	  (unsigned long int) grp->gr_gid);

  while (grp->gr_mem[i] != NULL)
    {
      fputs_unlocked (grp->gr_mem[i], stdout);
      ++i;
      if (grp->gr_mem[i] != NULL)
	putchar_unlocked (',');
    }
  putchar_unlocked ('\n');
}

static int
group_keys (int number, char *key[])
{
  int result = 0;
  int i;
  struct group *grp = NULL;

  if (number == 0)
    {
      setgrent ();
      while ((grp = getgrent ()) != NULL)
	print_group (grp);
      endgrent ();
      return result;
    }

  for (i = 0; i < number; ++i)
    {
      errno = 0;
      char *ep;
      gid_t arg_gid = strtoul(key[i], &ep, 10);

      if (errno != EINVAL && *key[i] != '\0' && *ep == '\0')
	/* Valid numeric gid.  */
	grp = getgrgid (arg_gid);
#ifdef __CYGWIN__
      /* Allow searching for Windows SIDs. */
      else
	{
	  BOOL try_as_cygname = FALSE;
	  PSID sid = get_sid_from_special_string (key[i], &try_as_cygname);

	  if (sid)
	    {
	      grp = (struct group *) cygwin_internal (CW_GETGRSID, 0, sid);
	      if (grp == (struct group *) -1) /* pre 1.7.29 Cygwin */
		grp = NULL;
	      LocalFree (sid);
	    }
	  else if (!try_as_cygname)
	    grp = getgrnam (key[i]);
	}
#else
      else
	grp = getgrnam (key[i]);
#endif

      if (grp == NULL)
	result = 2;
      else
	print_group (grp);
    }

  return result;
}

#ifndef __CYGWIN__
/* This is for gshadow */
static void
print_gshadow (struct sgrp *sg)
{
  unsigned int i = 0;

  printf ("%s:%s:",
	  sg->sg_namp ? sg->sg_namp : "",
	  sg->sg_passwd ? sg->sg_passwd : "");

  while (sg->sg_adm[i] != NULL)
    {
      fputs_unlocked (sg->sg_adm[i], stdout);
      ++i;
      if (sg->sg_adm[i] != NULL)
	putchar_unlocked (',');
    }

  putchar_unlocked (':');

  i = 0;
  while (sg->sg_mem[i] != NULL)
    {
      fputs_unlocked (sg->sg_mem[i], stdout);
      ++i;
      if (sg->sg_mem[i] != NULL)
	putchar_unlocked (',');
    }

  putchar_unlocked ('\n');
}

static int
gshadow_keys (int number, char *key[])
{
  int result = 0;
  int i;

  if (number == 0)
    {
      struct sgrp *sg;

      setsgent ();
      while ((sg = getsgent ()) != NULL)
	print_gshadow (sg);
      endsgent ();
      return result;
    }

  for (i = 0; i < number; ++i)
    {
      struct sgrp *sg;

      sg = getsgnam (key[i]);

      if (sg == NULL)
	result = 2;
      else
	print_gshadow (sg);
    }

  return result;
}
#endif /* !__CYGWIN__ */

/* This is for hosts */
static void
print_hosts (struct hostent *host)
{
  unsigned int cnt;

  for (cnt = 0; host->h_addr_list[cnt] != NULL; ++cnt)
    {
      char buf[INET6_ADDRSTRLEN];
      const char *ip = inet_ntop (host->h_addrtype, host->h_addr_list[cnt],
				  buf, sizeof (buf));

      printf ("%-15s %s", ip, host->h_name);

      unsigned int i;
      for (i = 0; host->h_aliases[i] != NULL; ++i)
	{
	  putchar_unlocked (' ');
	  fputs_unlocked (host->h_aliases[i], stdout);
	}
      putchar_unlocked ('\n');
    }
}

static int
hosts_keys (int number, char *key[])
{
  int result = 0;
  int i;
  struct hostent *host;

  if (number == 0)
    {
#ifdef __CYGWIN__
      fprintf (stderr, _("Enumeration not supported on %s\n"), "hosts");
      return 3;
#else
      sethostent (0);
      while ((host = gethostent ()) != NULL)
	print_hosts (host);
      endhostent ();
      return result;
#endif /* !__CYGWIN__ */
    }

  for (i = 0; i < number; ++i)
    {
      struct hostent *host = NULL;
      char addr[IN6ADDRSZ];

      if (inet_pton (AF_INET6, key[i], &addr) > 0)
	host = gethostbyaddr (addr, IN6ADDRSZ, AF_INET6);
      else if (inet_pton (AF_INET, key[i], &addr) > 0)
	host = gethostbyaddr (addr, INADDRSZ, AF_INET);
      else if ((host = gethostbyname2 (key[i], AF_INET6)) == NULL)
	host = gethostbyname2 (key[i], AF_INET);

      if (host == NULL)
	result = 2;
      else
	print_hosts (host);
    }

  return result;
}

/* This is for hosts, but using getaddrinfo */
static int
ahosts_keys_int (int af, int xflags, int number, char *key[])
{
  int result = 0;
  int i;
  struct hostent *host;

  if (number == 0)
    {
#ifdef __CYGWIN__
      fprintf (stderr, _("Enumeration not supported on %s\n"), "hosts");
      return 3;
#else
      sethostent (0);
      while ((host = gethostent ()) != NULL)
	print_hosts (host);
      endhostent ();
      return result;
#endif /* !__CYGWIN__ */
    }

  struct addrinfo hint;
  memset (&hint, '\0', sizeof (hint));
  hint.ai_flags = (AI_V4MAPPED | AI_ADDRCONFIG | AI_CANONNAME
		   | idn_flags | xflags);
  hint.ai_family = af;

  for (i = 0; i < number; ++i)
    {
      struct addrinfo *res;

      if (getaddrinfo (key[i], NULL, &hint, &res) != 0)
	result = 2;
      else
	{
	  struct addrinfo *runp = res;

	  while (runp != NULL)
	    {
	      char sockbuf[20];
	      const char *sockstr;
	      if (runp->ai_socktype == SOCK_STREAM)
		sockstr = "STREAM";
	      else if (runp->ai_socktype == SOCK_DGRAM)
		sockstr = "DGRAM";
	      else if (runp->ai_socktype == SOCK_RAW)
		sockstr = "RAW";
#ifdef SOCK_SEQPACKET
	      else if (runp->ai_socktype == SOCK_SEQPACKET)
		sockstr = "SEQPACKET";
#endif
#ifdef SOCK_RDM
	      else if (runp->ai_socktype == SOCK_RDM)
		sockstr = "RDM";
#endif
#ifdef SOCK_DCCP
	      else if (runp->ai_socktype == SOCK_DCCP)
		sockstr = "DCCP";
#endif
#ifdef SOCK_PACKET
	      else if (runp->ai_socktype == SOCK_PACKET)
		sockstr = "PACKET";
#endif
	      else
		{
		  snprintf (sockbuf, sizeof (sockbuf), "%d",
			    runp->ai_socktype);
		  sockstr = sockbuf;
		}

	      char buf[INET6_ADDRSTRLEN];
	      printf ("%-15s %-6s %s\n",
		      inet_ntop (runp->ai_family,
				 runp->ai_family == AF_INET
				 ? (void *) &((struct sockaddr_in *) runp->ai_addr)->sin_addr
				 : (void *) &((struct sockaddr_in6 *) runp->ai_addr)->sin6_addr,
				 buf, sizeof (buf)),
		      sockstr,
		      runp->ai_canonname ?: "");

	      runp = runp->ai_next;
	    }

	  freeaddrinfo (res);
	}
    }

  return result;
}

static int
ahosts_keys (int number, char *key[])
{
  return ahosts_keys_int (AF_UNSPEC, 0, number, key);
}

static int
ahostsv4_keys (int number, char *key[])
{
  return ahosts_keys_int (AF_INET, 0, number, key);
}

static int
ahostsv6_keys (int number, char *key[])
{
  return ahosts_keys_int (AF_INET6, AI_V4MAPPED, number, key);
}

#ifndef __CYGWIN__
/* This is for netgroup */
static int
netgroup_keys (int number, char *key[])
{
  int result = 0;

  if (number == 0)
    {
      fprintf (stderr, _("Enumeration not supported on %s\n"), "netgroup");
      return 3;
    }

  if (number == 4)
    {
      char *host = strcmp (key[1], "*") == 0 ? NULL : key[1];
      char *user = strcmp (key[2], "*") == 0 ? NULL : key[2];
      char *domain = strcmp (key[3], "*") == 0 ? NULL : key[3];

      printf ("%-21s (%s,%s,%s) = %d\n",
	      key[0], host ?: "", user ?: "", domain ?: "",
	      innetgr (key[0], host, user, domain));
    }
  else if (number == 1)
    {
      if (!setnetgrent (key[0]))
	result = 2;
      else
	{
	  char *p[3];

	  printf ("%-21s", key[0]);

	  while (getnetgrent (p, p + 1, p + 2))
	    printf (" (%s,%s,%s)", p[0] ?: " ", p[1] ?: "", p[2] ?: "");
	  putchar_unlocked ('\n');
	}
    }

  endnetgrent ();

  return result;
}
#endif /* !__CYGWIN__ */

/* This is for initgroups */
static int
initgroups_keys (int number, char *key[])
{
  int ngrps = 100;
  size_t grpslen = ngrps * sizeof (gid_t);
  gid_t *grps = alloca (grpslen);

  if (number == 0)
    {
      fprintf (stderr, _("Enumeration not supported on %s\n"), "initgroups");
      return 3;
    }

  for (int i = 0; i < number; ++i)
    {
      int no = ngrps;
      int n;
      while ((n = getgrouplist (key[i], -1, grps, &no)) == -1
	     && no > ngrps)
	{
	  grps = extend_alloca (grps, grpslen, no * sizeof (gid_t));
	  ngrps = no;
	}

      if (n == -1)
	return 1;

      printf ("%-21s", key[i]);
      for (int j = 0; j < n; ++j)
	if (grps[j] != -1)
	  printf (" %ld", (long int) grps[j]);
      putchar_unlocked ('\n');
    }

  return 0;
}

#ifndef __CYGWIN__
/* This is for networks */
static void
print_networks (struct netent *net)
{
  unsigned int i;
  struct in_addr ip;
  ip.s_addr = htonl (net->n_net);

  printf ("%-21s %s", net->n_name, inet_ntoa (ip));

  i = 0;
  while (net->n_aliases[i] != NULL)
    {
      putchar_unlocked (' ');
      fputs_unlocked (net->n_aliases[i], stdout);
      ++i;
    }
  putchar_unlocked ('\n');
}

static int
networks_keys (int number, char *key[])
{
  int result = 0;
  int i;
  struct netent *net;

  if (number == 0)
    {
      setnetent (0);
      while ((net = getnetent ()) != NULL)
	print_networks (net);
      endnetent ();
      return result;
    }

  for (i = 0; i < number; ++i)
    {
      if (isdigit (key[i][0]))
	net = getnetbyaddr (ntohl (inet_addr (key[i])), AF_UNSPEC);
      else
	net = getnetbyname (key[i]);

      if (net == NULL)
	result = 2;
      else
	print_networks (net);
    }

  return result;
}
#endif /* !__CYGWIN__ */

/* Now is all for passwd */
static void
print_passwd (struct passwd *pwd)
{
#ifdef __CYGWIN__
  /* Disable printing uid -1, the fallback invalid user called "????????" on
     Cygwin versions prior to the introduction of direct SAM/AD access. */
  if (pwd->pw_uid == (uid_t) -1)
    return;
  if (windows_only)
    {
      char *c = strstr (pwd->pw_gecos, "S-1-");
      print_windows_mapping (pwd->pw_name, (unsigned long) pwd->pw_uid, c);
      return;
    }
#endif
  printf ("%s:%s:%lu:%lu:%s:%s:%s\n",
	  pwd->pw_name ? pwd->pw_name : "",
	  pwd->pw_passwd ? pwd->pw_passwd : "",
	  (unsigned long int) pwd->pw_uid,
	  (unsigned long int) pwd->pw_gid,
	  pwd->pw_gecos ? pwd->pw_gecos : "",
	  pwd->pw_dir ? pwd->pw_dir : "",
	  pwd->pw_shell ? pwd->pw_shell : "");
}

static int
passwd_keys (int number, char *key[])
{
  int result = 0;
  int i;
  struct passwd *pwd = NULL;

  if (number == 0)
    {
      setpwent ();
      while ((pwd = getpwent ()) != NULL)
	print_passwd (pwd);
      endpwent ();
      return result;
    }

  for (i = 0; i < number; ++i)
    {
      errno = 0;
      char *ep;
      uid_t arg_uid = strtoul(key[i], &ep, 10);

      if (errno != EINVAL && *key[i] != '\0' && *ep == '\0')
	/* Valid numeric uid.  */
	pwd = getpwuid (arg_uid);
#ifdef __CYGWIN__
      /* Allow searching for Windows SIDs. */
      else
	{
	  BOOL try_as_cygname = FALSE;
	  PSID sid = get_sid_from_special_string (key[i], &try_as_cygname);

	  if (sid)
	    {
	      pwd = (struct passwd *) cygwin_internal (CW_GETPWSID, 0, sid);
	      if (pwd == (struct passwd *) -1) /* pre 1.7.29 Cygwin */
		pwd = NULL;
	      LocalFree (sid);
	    }
	  else if (!try_as_cygname)
	    pwd = getpwnam (key[i]);
	}
#else
      else
	pwd = getpwnam (key[i]);
#endif

      if (pwd == NULL)
	result = 2;
      else
	print_passwd (pwd);
    }

  return result;
}

/* This is for protocols */
static void
print_protocols (struct protoent *proto)
{
  unsigned int i;

  printf ("%-21s %d", proto->p_name, proto->p_proto);

  i = 0;
  while (proto->p_aliases[i] != NULL)
    {
      putchar_unlocked (' ');
      fputs_unlocked (proto->p_aliases[i], stdout);
      ++i;
    }
  putchar_unlocked ('\n');
}

static int
protocols_keys (int number, char *key[])
{
  int result = 0;
  int i;
  struct protoent *proto;

  if (number == 0)
    {
      setprotoent (0);
      while ((proto = getprotoent ()) != NULL)
	print_protocols (proto);
      endprotoent ();
      return result;
    }

  for (i = 0; i < number; ++i)
    {
      if (isdigit (key[i][0]))
	proto = getprotobynumber (atol (key[i]));
      else
	proto = getprotobyname (key[i]);

      if (proto == NULL)
	result = 2;
      else
	print_protocols (proto);
    }

  return result;
}

#ifndef __CYGWIN__
/* Now is all for rpc */
static void
print_rpc (struct rpcent *rpc)
{
  int i;

  printf ("%-15s %d%s",
	  rpc->r_name, rpc->r_number, rpc->r_aliases[0] ? " " : "");

  for (i = 0; rpc->r_aliases[i]; ++i)
    printf (" %s", rpc->r_aliases[i]);
  putchar_unlocked ('\n');
}

static int
rpc_keys (int number, char *key[])
{
  int result = 0;
  int i;
  struct rpcent *rpc;

  if (number == 0)
    {
      setrpcent (0);
      while ((rpc = getrpcent ()) != NULL)
	print_rpc (rpc);
      endrpcent ();
      return result;
    }

  for (i = 0; i < number; ++i)
    {
      if (isdigit (key[i][0]))
	rpc = getrpcbynumber (atol (key[i]));
      else
	rpc = getrpcbyname (key[i]);

      if (rpc == NULL)
	result = 2;
      else
	print_rpc (rpc);
    }

  return result;
}
#endif /* !__CYGWIN__ */

/* for services */
static void
print_services (struct servent *serv)
{
  unsigned int i;

  printf ("%-21s %d/%s", serv->s_name, ntohs (serv->s_port), serv->s_proto);

  i = 0;
  while (serv->s_aliases[i] != NULL)
    {
      putchar_unlocked (' ');
      fputs_unlocked (serv->s_aliases[i], stdout);
      ++i;
    }
  putchar_unlocked ('\n');
}

static int
services_keys (int number, char *key[])
{
  int result = 0;
  int i;
  struct servent *serv;

  if (!number)
    {
      setservent (0);
      while ((serv = getservent ()) != NULL)
	print_services (serv);
      endservent ();
      return result;
    }

  for (i = 0; i < number; ++i)
    {
      struct servent *serv;
      char *proto = strchr (key[i], '/');

      if (proto != NULL)
	*proto++ = '\0';

      char *endptr;
      long port = strtol (key[i], &endptr, 10);

      if (isdigit (key[i][0]) && *endptr == '\0'
	  && 0 <= port && port <= 65535)
	serv = getservbyport (htons (port), proto);
      else
	serv = getservbyname (key[i], proto);

      if (serv == NULL)
	result = 2;
      else
	print_services (serv);
    }

  return result;
}

#ifndef __CYGWIN__
/* This is for shadow */
static void
print_shadow (struct spwd *sp)
{
  printf ("%s:%s:",
	  sp->sp_namp ? sp->sp_namp : "",
	  sp->sp_pwdp ? sp->sp_pwdp : "");

#define SHADOW_FIELD(n) \
  if (sp->n == -1)							      \
    putchar_unlocked (':');						      \
  else									      \
    printf ("%ld:", sp->n)

  SHADOW_FIELD (sp_lstchg);
  SHADOW_FIELD (sp_min);
  SHADOW_FIELD (sp_max);
  SHADOW_FIELD (sp_warn);
  SHADOW_FIELD (sp_inact);
  SHADOW_FIELD (sp_expire);
  if (sp->sp_flag == ~0ul)
    putchar_unlocked ('\n');
  else
    printf ("%lu\n", sp->sp_flag);
}

static int
shadow_keys (int number, char *key[])
{
  int result = 0;
  int i;

  if (number == 0)
    {
      struct spwd *sp;

      setspent ();
      while ((sp = getspent ()) != NULL)
	print_shadow (sp);
      endspent ();
      return result;
    }

  for (i = 0; i < number; ++i)
    {
      struct spwd *sp;

      sp = getspnam (key[i]);

      if (sp == NULL)
	result = 2;
      else
	print_shadow (sp);
    }

  return result;
}
#endif /* !__CYGWIN__ */

struct
  {
    const char *name;
    int (*func) (int number, char *key[]);
  } databases[] =
  {
#define D(name) { #name, name ## _keys },
D(ahosts)
D(ahostsv4)
D(ahostsv6)
#ifndef __CYGWIN__
D(aliases)
D(ethers)
#endif /* !__CYGWIN__ */
D(group)
#ifndef __CYGWIN__
D(gshadow)
#endif /* !__CYGWIN__ */
D(hosts)
D(initgroups)
#ifndef __CYGWIN__
D(netgroup)
D(networks)
#endif /* !__CYGWIN__ */
D(passwd)
D(protocols)
#ifndef __CYGWIN__
D(rpc)
#endif /* !__CYGWIN__ */
D(services)
#ifndef __CYGWIN__
D(shadow)
#endif /* !__CYGWIN__ */
#undef D
    { NULL, NULL }
  };

/* Handle arguments found by argp. */
static error_t
parse_option (int key, char *arg, struct argp_state *state)
{
  char *endp;
  switch (key)
    {
    case 's':
#ifndef __CYGWIN__
      endp = strchr (arg, ':');
      if (endp == NULL)
	/* No specific database, change them all.  */
	for (int i = 0; databases[i].name != NULL; ++i)
	  __nss_configure_lookup (databases[i].name, arg);
      else
	{
	  int i;
	  for (i = 0; databases[i].name != NULL; ++i)
	    if (strncmp (databases[i].name, arg, endp - arg) == 0)
	      {
		__nss_configure_lookup (databases[i].name, endp + 1);
		break;
	      }
	  if (databases[i].name == NULL)
	    error (EXIT_FAILURE, 0, gettext ("Unknown database name"));
	}
#endif /* !__CYGWIN__ */
      break;

    case 'i':
      idn_flags = 0;
      break;

#ifdef __CYGWIN__
    case 'w':
      windows_only = 1;
      break;
#endif

    default:
      return ARGP_ERR_UNKNOWN;
    }

  return 0;
}


static char *
more_help (int key, const char *text, void *input)
{
  switch (key)
    {
      size_t len;
      char *doc;
      FILE *fp;

    case ARGP_KEY_HELP_EXTRA:
      /* We print some extra information.  */
      fp = open_memstream (&doc, &len);
      if (fp != NULL)
	{
	  fputs_unlocked (_("Supported databases:\n"), fp);

	  for (int i = 0, col = 0; databases[i].name != NULL; ++i)
	    {
	      len = strlen (databases[i].name);
	      if (i != 0)
		{
		  if (col + len > 72)
		    {
		      col = 0;
		      fputc_unlocked ('\n', fp);
		    }
		  else
		    fputc_unlocked (' ', fp);
		}

	      fputs_unlocked (databases[i].name, fp);
	      col += len + 1;
	    }

	  fputs ("\n\n", fp);

	  fprintf (fp, gettext ("\
For bug reporting instructions, please see:\n\
%s.\n"), REPORT_BUGS_TO);

	  if (fclose (fp) == 0)
	    return doc;
	}
      break;

    default:
      break;
    }
  return (char *) text;
}


/* the main function */
int
main (int argc, char *argv[])
{
#ifndef __CYGWIN__
  /* Debugging support.  */
  mtrace ();
#endif /* !__CYGWIN__ */

  /* Set locale via LC_ALL.  */
  setlocale (LC_ALL, "");
  /* Set the text message domain.  */
  textdomain (PACKAGE);

  /* Parse and process arguments.  */
  int remaining;
  argp_parse (&argp, argc, argv, 0, &remaining, NULL);

  if ((argc - remaining) < 1)
    {
      error (0, 0, gettext ("wrong number of arguments"));
      argp_help (&argp, stdout, ARGP_HELP_SEE, program_invocation_short_name);
      return 1;
    }

  for (int i = 0; databases[i].name; ++i)
    if (argv[remaining][0] == databases[i].name[0]
	&& !strcmp (argv[remaining], databases[i].name))
      return databases[i].func (argc - remaining - 1, &argv[remaining + 1]);

  fprintf (stderr, _("Unknown database: %s\n"), argv[remaining]);
  argp_help (&argp, stdout, ARGP_HELP_SEE, program_invocation_short_name);
  return 1;
}
